/**
* \file: VirtualSensorSource.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: M. Adachi / ADITJ/SW / madachi@jp.adit-jv.com
*
* \copyright (c) 2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <adit_logging.h>
#include "VirtualSensorSource.h"
#include "Sensors.h"
#include <limits.h>

#include <inttypes.h>

LOG_DECLARE_CONTEXT(aauto_sensor);

#ifndef COMP_GIT_VERSION
#define COMP_GIT_VERSION "aauto_r3_d01 or newer"
#endif

namespace adit { namespace aauto {

VirtualSensorSource::VirtualSensorSource(uint8_t inSessionId, MessageRouter* inMessageRouter, uint32_t inLocationCharacterization)
{
    threadId = 0;
    threadRunning = false;
    previousTime = 0;
    testLocate = "";
    loopTime = 0;
    verbose = false;
    
    LOG_REGISTER_CONTEXT(aauto_sensor, "TVSN", "Test aauto virtual sensor");
    LOGD_DEBUG((aauto_sensor, "aauto-test-v-sensor git tag: %s", COMP_GIT_VERSION));

    if(inMessageRouter == nullptr)
    {
        LOG_ERROR((aauto_sensor, "MessageRouter pointer is null for VirtualSensorSource."));
        return;
    }

    sensor = ::shared_ptr<SensorSource> (new SensorSource(inSessionId, inMessageRouter, inLocationCharacterization));
    if(sensor == nullptr)
    {
        LOG_ERROR((aauto_sensor, "Could not allocate resource for SensorSource."));
        return;
    }
}

VirtualSensorSource::~VirtualSensorSource()
{
    LOG_UNREGISTER_CONTEXT(aauto_sensor);
}


bool VirtualSensorSource::StartThread()
{
    if (threadId != 0)
    {
        LOG_ERROR((aauto_sensor, "SensorSource thread is already running"));
        return false;
    }

    // thread start
    int ret = pthread_create(&threadId, NULL, sensorThread, this);
    if (ret != 0)
    {
        LOG_ERROR((aauto_sensor, "pthread_create failed: %d %s", errno, strerror(errno)));
    }

    return true;
}

void VirtualSensorSource::StopThread()
{
    if(threadRunning)
    {
        threadRunning = false;

        if(threadId != 0)
        {
            // wait for thread stop
            int ret = pthread_join(threadId, nullptr);
            if (ret != 0)
            {
                // if pthread_join fails just log the error using errno
                LOG_ERROR((aauto_sensor, "pthread_join failed with error %d %s", errno, strerror(errno)));
            }
            else
            {
                threadId = 0;
            }
        }
        else
        {
            LOG_ERROR((aauto_sensor, "SensorSource thread not created"));
        }
    }
    else
    {
        LOG_INFO((aauto_sensor, "SensorSource thread not running. Was already stopped"));
    }
}

bool VirtualSensorSource::init()
{
    bool ret = true;

    if(SetSensorType())
    {
        loopTime = getConfig("test-loop-time", 1000);
        verbose = 0 != getConfig("enable-verbose-logging", 0);
        testLocate = getConfig("test-location", "");

        if(!StartThread())
        {
            ret = false;
            LOG_ERROR((aauto_sensor, "Couldn't start thread for VirtualSensor"));
        }
    }
    else
    {
        ret = false;
        LOG_ERROR((aauto_sensor, "Couldn't set the SensorType"));
    }

    return ret;
}

void VirtualSensorSource::setMode(SensorType type, sensorTestMode mode)
{
    for(std::list<::shared_ptr<SensorTest>>::iterator it = sensorTests.begin(); it != sensorTests.end(); it++)
    {
        if((*it)->getType() == type)
        {
            LOGD_DEBUG((aauto_sensor, "Change Sensor mode.(%d)", mode));
            (*it)->setMode(mode);
            break;
        }
    }
}

void VirtualSensorSource::stop()
{
    StopThread();
}

void VirtualSensorSource::setConfigItem(std::string inKey, std::string inValue)
{
    config.insert(std::pair<std::string, std::string>(inKey, inValue));
}

bool VirtualSensorSource::SetSensorType()
{
    sensorTable sensorTable[] = {
        // configuration key                    sensor type code                time(ms)  count
        {"enable-sensor-location"               ,SENSOR_LOCATION                , 500,     -1},
        {"enable-sensor-compass"                ,SENSOR_COMPASS                 , 100,      0},
        {"enable-sensor-speed"                  ,SENSOR_SPEED                   , 100,      0},
        {"enable-sensor-rpm"                    ,SENSOR_RPM                     , 100,      0},
        {"enable-sensor-odometer"               ,SENSOR_ODOMETER                , 1000,     0},
        {"enable-sensor-fuel"                   ,SENSOR_FUEL                    , 3000,     0},
        {"enable-sensor-parking-brake"          ,SENSOR_PARKING_BRAKE           , 2000,     0},
        {"enable-sensor-gear"                   ,SENSOR_GEAR                    , 1000,     0},
        {"enable-sensor-obdii-diagnostic-code"  ,SENSOR_OBDII_DIAGNOSTIC_CODE   , 0,        0},
        {"enable-sensor-night-mode"             ,SENSOR_NIGHT_MODE              , 3000,     3},
        {"enable-sensor-environment-data"       ,SENSOR_ENVIRONMENT_DATA        , 0,        0},
        {"enable-sensor-hvac-data"              ,SENSOR_HVAC_DATA               , 0,        0},
        {"enable-sensor-driving-status-data"    ,SENSOR_DRIVING_STATUS_DATA     , 1000,     0},
        {"enable-sensor-dead-reckoning-data"    ,SENSOR_DEAD_RECKONING_DATA     , 0,        0},
        {"enable-sensor-passenger-data"         ,SENSOR_PASSENGER_DATA          , 0,        0},
        {"enable-sensor-door-data"              ,SENSOR_DOOR_DATA               , 1000,     0},
        {"enable-sensor-light-data"             ,SENSOR_LIGHT_DATA              , 0,        0},
        {"enable-sensor-tire-pressure-data"     ,SENSOR_TIRE_PRESSURE_DATA      , 0,        0},
        {"enable-sensor-accelerometer-data"     ,SENSOR_ACCELEROMETER_DATA      , 0,        0},
        {"enable-sensor-gyroscope-data"         ,SENSOR_GYROSCOPE_DATA          , 0,        0},
        {"enable-sensor-gps-satellite-data"     ,SENSOR_GPS_SATELLITE_DATA      , 0,        0}
    };

    bool ret = false;
    int max = sizeof(sensorTable) / sizeof(sensorTable[0]);
    
    for(int ix = 0; ix < max; ix++)
    {
        if(getConfig(sensorTable[ix].cfgstr, 0))
        {
            LOGD_DEBUG((aauto_sensor, "sensor->registerSensor: %s ", sensorTable[ix].cfgstr.c_str()));
            sensor->registerSensor(sensorTable[ix].type);
            sensorList.push_back(sensorTable[ix]);
            ret = true;
        }
    }

    return ret;
}

std::string VirtualSensorSource::getConfig(const std::string& key, std::string defaultValue) const
{
    std::string  result = defaultValue;
    std::string  item;

    auto found = config.find(key);
    if(found != config.end())
    {
        result = found->second;
    }

    return result;
}

int64_t VirtualSensorSource::getConfig(const std::string& key, int64_t defaultValue) const
{
    int64_t result = defaultValue;
    std::string  item;

    auto found = config.find(key);
    if(found != config.end())
    {
        char* p = nullptr;
        int64_t converted = strtoll((found->second).c_str(), &p, 0);
        if ((converted == LLONG_MAX || converted == LLONG_MIN) && errno == ERANGE)
        {
            LOG_WARN((aauto_sensor, "config item %s is out of range, default to %" PRId64 "", key.c_str(), result));
        }
        else if (p != nullptr && *p != '\0')
        {
            LOG_WARN((aauto_sensor, "config item %s is not a number, default to %" PRId64 "", key.c_str(), result));
        }
        else
        {
            result = converted;
        }
    }

    return result;
}

void* VirtualSensorSource::sensorThread(void* inData)
{
    LOGD_DEBUG((aauto_sensor, "Sensor source report thread started"));

    VirtualSensorSource* me = static_cast<VirtualSensorSource*>(inData);

    // set thread name
    prctl(PR_SET_NAME, "SensorSource", 0, 0, 0);

    if(me->prepareTest())
    {
        int err_code = me->start();
        if(err_code != STATUS_SUCCESS)
        {
            LOG_ERROR((aauto_sensor, "Error happened in sensor test so test has finished.(%d)", err_code));
        }
    }
    else
    {
        LOG_ERROR((aauto_sensor, "Error prepare of sensor test."));
    }

    LOGD_DEBUG((aauto_sensor, "Sensor source report thread ended"));

    return nullptr;
}

bool VirtualSensorSource::prepareTest()
{
    bool ret = true;
    ::shared_ptr<SensorTest> sensorTest;

    // Loops sensorType-list until end of list.
    for(std::list<sensorTable>::iterator it = sensorList.begin(); it != sensorList.end(); it++)
    {
        switch(it->type)
        {
        case SENSOR_NIGHT_MODE:
            sensorTest = new NightModeData(verbose, *it);
            break;

        case SENSOR_SPEED:
            sensorTest = new SpeedData(verbose, *it);
            break;

        case SENSOR_COMPASS:
            sensorTest = new CompassData(verbose, *it);
            break;

        case SENSOR_RPM:
            sensorTest = new RPMData(verbose, *it);
            break;
        
        case SENSOR_FUEL:
            sensorTest = new FuelData(verbose, *it);
            break;

        case SENSOR_ODOMETER:
            sensorTest = new OdometerData(verbose, *it);
            break;
            
        case SENSOR_PARKING_BRAKE:
            sensorTest = new ParkingBrakeData(verbose, *it);
            break;
            
        case SENSOR_GEAR:
            sensorTest = new GearData(verbose, *it);
            break;
            
        case SENSOR_DOOR_DATA:
            sensorTest = new DoorData(verbose, *it);
            break;

        case SENSOR_DRIVING_STATUS_DATA:
            sensorTest = new DrivingStatusData(verbose, *it);
            break;

        case SENSOR_LOCATION:
            /* do not create until auto app is no longer crashing on the mobile device */
            sensorTest = new LocationData(verbose, *it, testLocate);
            break;

        default:
            sensorTest = nullptr;
            LOG_ERROR((aauto_sensor, "Error: There is not type of sensor."));
            break;
        }

        // Push sensorTable to list of   
        if(sensorTest != nullptr)
        {
            sensorTests.push_back(sensorTest);
        }
    }
    
    return ret;
}

void VirtualSensorSource::wait_us(uint32_t usec)
{
    struct timespec  req;

    req.tv_sec = usec / 1000000;            // us -> s
    req.tv_nsec = (usec % 1000000) * 1000;  // us -> ns

    nanosleep(&req, 0);
}

void VirtualSensorSource::getAdjustTime()
{
    struct timeval pretime;

    gettimeofday( &pretime, NULL );
    previousTime = (pretime.tv_sec * 1000000 + pretime.tv_usec);
}

void VirtualSensorSource::autoAdjustTime()
{
    struct timeval  timeval; 
    uint32_t        worktime;

    gettimeofday( &timeval, NULL );
    worktime = (timeval.tv_sec * 1000000 + timeval.tv_usec);

    if(loopTime > (worktime - previousTime))
    {
        wait_us(loopTime - (worktime - previousTime));
    }
}

int VirtualSensorSource::start()
{
    int err_code = STATUS_SUCCESS;
    threadRunning = true;

    // test loop
    while(threadRunning && (err_code == STATUS_SUCCESS))
    {
        getAdjustTime();
        // Loops each sensorType until end of list.
        for(std::list<::shared_ptr<SensorTest>>::iterator it = sensorTests.begin(); it != sensorTests.end(); it++)
        {
            if((err_code = (*it)->start(sensor)) != STATUS_SUCCESS)
            {
                LOG_ERROR((aauto_sensor, "Error: Sensor test fail."));
                break;
            }
        }
        autoAdjustTime();
    }

    return err_code;
}

} } // namespace adit { namespace aauto {
